"use client";
import { useEffect, useMemo, useState, useRef } from "react";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";
import {
  ArrowLeft,
  Plus,
  Pencil,
  Trash2,
  Check,
  X,
  Loader2,
  Eye,
  EyeOff,
  Search,
  Users,
  ChevronDown,
  ChevronRight,
  QrCode,
  Download,
} from "lucide-react";
import { ConfirmDialog } from "@/components/ConfirmDialog";
import QRCodeGenerator from "@/components/QRCodeGenerator";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";

/* Types */
type Department = { id: string; name: string; isVisible: boolean };
type LocationItem = {
  id: string;
  name: string;
  departmentId: string;
  isVisible: boolean;
};
type ElementItem = {
  id: string;
  name: string;
  locationId: string;
  isVisible: boolean;
};
type Role = {
  id: string;
  name: string;
  description?: string | null;
  isActive: boolean;
};
type ElementRole = {
  id: string;
  name: string;
  description?: string | null;
  assignedAt: string;
};

export default function CatalogsAdminPage() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const role = String((session?.user as any)?.role || "").toUpperCase();
  const isAdmin = role === "ADMIN";

  const [deps, setDeps] = useState<Department[]>([]);
  const [depId, setDepId] = useState<string>("");
  const [locations, setLocations] = useState<LocationItem[]>([]);
  const [locationId, setLocationId] = useState<string>("");
  const [elements, setElements] = useState<ElementItem[]>([]);
  const [roles, setRoles] = useState<Role[]>([]);
  const [expandedElements, setExpandedElements] = useState<Set<string>>(
    new Set()
  );
  const [depName, setDepName] = useState("");
  const [locName, setLocName] = useState("");
  const [elName, setElName] = useState("");
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);
  const [msg, setMsg] = useState<string | null>(null);
  const [dialogError, setDialogError] = useState<string | null>(null);
  const [addDeptOpen, setAddDeptOpen] = useState(false);
  const [addLocationOpen, setAddLocationOpen] = useState(false);
  const [addElementOpen, setAddElementOpen] = useState(false);
  const [confirmOpen, setConfirmOpen] = useState(false);
  const [confirmAction, setConfirmAction] = useState<null | (() => void)>(null);
  const [showHidden, setShowHidden] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [locationSearchTerm, setLocationSearchTerm] = useState("");
  const [elementSearchTerm, setElementSearchTerm] = useState("");
  const [showQRCode, setShowQRCode] = useState<string | null>(null);
  const [qrCodeType, setQrCodeType] = useState<
    "department" | "location" | "element"
  >("department");

  const currentDep = useMemo(
    () => deps.find((d) => d.id === depId) || null,
    [deps, depId]
  );
  const currentLoc = useMemo(
    () => locations.find((l) => l.id === locationId) || null,
    [locations, locationId]
  );

  const generateQRUrl = (
    id: string,
    type: "department" | "location" | "element" = "department",
    departmentId?: string,
    locationId?: string
  ) => {
    const baseUrl = typeof window !== "undefined" ? window.location.origin : "";
    if (type === "element") {
      const params = new URLSearchParams({ elementId: id });
      if (locationId) {
        params.append("locationId", locationId);
      }
      if (departmentId) {
        params.append("departmentId", departmentId);
      }
      return `${baseUrl}/tickets/new?${params.toString()}`;
    }
    if (type === "location") {
      const params = new URLSearchParams({ locationId: id });
      if (departmentId) {
        params.append("departmentId", departmentId);
      }
      return `${baseUrl}/tickets/new?${params.toString()}`;
    }
    return `${baseUrl}/tickets/new?department=${id}`;
  };

  const downloadQRCode = (
    id: string,
    name: string,
    type: "department" | "location" | "element" = "department"
  ) => {
    const modal = document.querySelector('[data-modal="qr-code"]');
    const canvas = modal?.querySelector("canvas") as HTMLCanvasElement;
    if (canvas) {
      const link = document.createElement("a");
      link.download = `qr-code-${name.replace(/\s+/g, "-").toLowerCase()}.png`;
      link.href = canvas.toDataURL();
      link.click();
    }
  };

  async function loadDeps() {
    setLoading(true);
    setErr(null);
    try {
      const params = new URLSearchParams();
      if (showHidden) params.append("showHidden", "true");
      if (searchTerm) params.append("search", searchTerm);
      const url = `/api/departments${
        params.toString() ? `?${params.toString()}` : ""
      }`;
      const r = await fetch(url);
      const j = await r.json();
      if (!j.ok) throw new Error(j.error || "Failed to load departments");
      setDeps(j.items || []);
      if (!depId && j.items?.length) setDepId(j.items[0].id);
      setLoading(false);
    } catch (e: any) {
      setErr(String(e?.message || e));
      setLoading(false);
    }
  }

  async function loadLocations(did = depId) {
    if (!did) {
      setLocations([]);
      setLocationId("");
      return;
    }
    try {
      const params = new URLSearchParams();
      params.append("departmentId", did);
      if (showHidden) params.append("showHidden", "true");
      if (locationSearchTerm) params.append("search", locationSearchTerm);
      const url = `/api/locations?${params.toString()}`;
      const r = await fetch(url);
      const j = await r.json();
      if (!j.ok) throw new Error(j.error || "Failed to load locations");
      setLocations(j.items || []);
      if (!locationId && j.items?.length) setLocationId(j.items[0].id);
    } catch (e: any) {
      setErr(String(e?.message || e));
    }
  }

  async function loadElements(lid = locationId) {
    if (!lid) {
      setElements([]);
      return;
    }
    try {
      const params = new URLSearchParams();
      params.append("locationId", lid);
      if (showHidden) params.append("showHidden", "true");
      if (elementSearchTerm) params.append("search", elementSearchTerm);
      const url = `/api/elements?${params.toString()}`;
      const r = await fetch(url);
      const j = await r.json();
      if (!j.ok) throw new Error(j.error || "Failed to load elements");
      setElements(j.items || []);
    } catch (e: any) {
      setErr(String(e?.message || e));
    }
  }

  async function loadRoles() {
    try {
      const r = await fetch("/api/admin/roles");
      const j = await r.json();
      if (!j.ok) throw new Error(j.error || "Failed to load roles");
      setRoles(j.roles || []);
    } catch (e) {
      console.error("Failed to load roles:", e);
    }
  }

  async function handleAddDepartment() {
    if (!depName.trim()) return;
    try {
      const r = await fetch("/api/departments", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ name: depName.trim() }),
      });
      const j = await r.json();
      if (!j.ok) {
        setErr(j.error || "Create failed");
        return;
      }
      setDepName("");
      setAddDeptOpen(false);
      await loadDeps();
      if (j.item?.id) setDepId(j.item.id);
      setMsg("Department created successfully.");
    } catch (e: any) {
      setErr(e?.message || "Create failed");
    }
  }

  async function handleAddLocation() {
    if (!depId || !locName.trim()) return;
    try {
      const r = await fetch("/api/locations", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: locName.trim(),
          departmentId: depId,
        }),
      });
      const j = await r.json();
      if (!j.ok) {
        setErr(j.error || "Create failed");
        return;
      }
      setLocName("");
      setAddLocationOpen(false);
      await loadLocations(depId);
      if (j.item?.id) setLocationId(j.item.id);
      setMsg("Location created successfully.");
    } catch (e: any) {
      setErr(e?.message || "Create failed");
    }
  }

  async function handleAddElement() {
    if (!locationId || !elName.trim()) return;
    try {
      const r = await fetch("/api/elements", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name: elName.trim(),
          locationId: locationId,
        }),
      });
      const j = await r.json();
      if (!j.ok) {
        setErr(j.error || "Create failed");
        return;
      }
      setElName("");
      setAddElementOpen(false);
      await loadElements(locationId);
      setMsg("Equipment created successfully.");
    } catch (e: any) {
      setErr(e?.message || "Create failed");
    }
  }

  useEffect(() => {
    loadDeps();
    loadRoles();
  }, [showHidden, searchTerm]);

  useEffect(() => {
    loadLocations(depId);
  }, [depId, showHidden, locationSearchTerm]);

  useEffect(() => {
    loadElements(locationId);
  }, [locationId, showHidden, elementSearchTerm]);

  if (status === "loading")
    return (
      <main className='max-w-5xl mx-auto px-4 py-6'>{t("common.loading")}</main>
    );

  if (!isAdmin)
    return (
      <main className='max-w-5xl mx-auto px-4 py-6'>
        <h1 className='text-2xl font-bold'>{t("admin.catalogs")}</h1>
        <div className='alert alert-error mt-3 text-sm sm:text-base'>
          <Trash2 />
          <div>
            <p className='font-semibold'>{t("admin.forbidden")}</p>
            <p className='opacity-90'>{t("admin.noPermission")}</p>
          </div>
        </div>
      </main>
    );

  return (
    <main className='max-w-5xl mx-auto px-4 sm:px-6 py-6 space-y-6'>
      {/* Header */}
      <header className='flex flex-col sm:flex-row sm:items-center gap-3 mb-6'>
        <Link
          href='/'
          className='btn-secondary inline-flex items-center gap-2 w-full sm:w-auto'
        >
          <ArrowLeft className='w-4 h-4' />
          {t("common.home")}
        </Link>
        <h1 className='text-2xl font-bold ml-1 text-foreground text-center sm:text-left'>
          {t("admin.catalogs")} (
          {t("admin.departmentsAndElements") || "Departments & Elements"})
        </h1>
        <div className='sm:ml-auto'>
          <Link
            href='/admin/technicians'
            className='btn-secondary w-full sm:w-auto'
          >
            {t("admin.technicians")}
          </Link>
        </div>
      </header>

      {err && (
        <div className='alert alert-error mb-4 text-sm sm:text-base'>
          <X />
          <div>
            <p className='font-semibold'>{t("common.error")}</p>
            <p className='opacity-90'>{err}</p>
          </div>
        </div>
      )}

      {msg && (
        <div className='alert alert-success mb-4 text-sm sm:text-base'>
          <Check />
          <div>
            <p className='font-semibold'>{t("common.success")}</p>
            <p className='opacity-90'>{msg}</p>
          </div>
        </div>
      )}

      {/* Departments */}
      <section className='card p-5 mb-6'>
        <div className='flex flex-col sm:flex-row sm:items-center justify-between gap-3'>
          <h2 className='text-lg font-semibold'>
            {t("admin.departments") || "Departments"}
          </h2>
          <label className='flex items-center gap-2 text-sm'>
            <input
              type='checkbox'
              checked={showHidden}
              onChange={(e) => setShowHidden(e.target.checked)}
              className='rounded'
            />
            {t("admin.showDisabledItems") || "Show disabled items"}
          </label>
        </div>

        {/* Search departments */}
        <div className='mt-6'>
          <div className='relative max-w-sm w-full'>
            <Search className='absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground z-10' />
            <Input
              type='text'
              placeholder={t("admin.searchDepartments")}
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className='w-full pl-10'
            />
          </div>
        </div>

        {/* Add department */}
        <div className='mt-6'>
          <Dialog open={addDeptOpen} onOpenChange={setAddDeptOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className='w-4 h-4' />
                {t("admin.addNewDepartment")}
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>{t("admin.addNewDepartment")}</DialogTitle>
              </DialogHeader>
              <div className='py-4'>
                <div className='space-y-4'>
                  <div className='space-y-2'>
                    <Label htmlFor='depName'>{t("admin.departmentName")}</Label>
                    <Input
                      id='depName'
                      placeholder={t("admin.enterDepartmentName")}
                      value={depName}
                      onChange={(e) => setDepName(e.target.value)}
                      onKeyPress={(e) => {
                        if (e.key === "Enter") {
                          e.preventDefault();
                          handleAddDepartment();
                        }
                      }}
                    />
                  </div>
                  <div className='flex justify-end gap-2'>
                    <Button
                      variant='outline'
                      onClick={() => {
                        setAddDeptOpen(false);
                        setDepName("");
                      }}
                    >
                      {t("common.cancel")}
                    </Button>
                    <Button
                      onClick={handleAddDepartment}
                      disabled={!depName.trim()}
                    >
                      {t("admin.addNewDepartment")}
                    </Button>
                  </div>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>

        {/* List */}
        {loading ? (
          <div className='mt-6 flex items-center justify-center gap-2 text-muted-foreground py-8'>
            <Loader2 className='w-5 h-5 animate-spin' />
            <span>{t("admin.loadingDepartments")}</span>
          </div>
        ) : deps.length === 0 ? (
          <div className='mt-6 text-center py-12 bg-muted/20 rounded-lg border border-dashed'>
            <div className='w-12 h-12 mx-auto mb-4 bg-muted rounded-full flex items-center justify-center'>
              <Plus className='w-6 h-6 text-muted-foreground' />
            </div>
            <h3 className='text-lg font-medium mb-2'>
              {t("admin.noDepartmentsYet")}
            </h3>
            <p className='text-muted-foreground'>
              {t("admin.createFirstDepartment")}
            </p>
          </div>
        ) : (
          <div className='mt-6'>
            <div className='flex flex-col sm:flex-row sm:items-center justify-between mb-4 gap-2'>
              <h3 className='text-lg font-semibold'>
                {t("admin.departmentsWithCount")} ({deps.length})
              </h3>
              <p className='text-sm text-muted-foreground text-center sm:text-right'>
                {t("ticketDetail.tapDepartmentToManage") ||
                  "Tap a department to manage its elements"}
              </p>
            </div>
            <div className='grid grid-cols-1 sm:grid-cols-2 xl:grid-cols-3 gap-4 sm:gap-6'>
              {deps.map((d) => (
                <DepartmentRow
                  key={d.id}
                  dept={d}
                  selected={depId === d.id}
                  onSelect={() => setDepId(d.id)}
                  onRename={async (newName) => {
                    const r = await fetch(`/api/departments/${d.id}/update`, {
                      method: "PATCH",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({ name: newName }),
                    });
                    const j = await r.json();
                    if (!j.ok) {
                      setErr(j.error || "Update failed");
                      return;
                    }
                    await loadDeps();
                  }}
                  onToggle={async () => {
                    const r = await fetch(`/api/departments/${d.id}/toggle`, {
                      method: "PATCH",
                    });
                    const j = await r.json();
                    if (!j.ok) {
                      setErr(j.error || "Toggle failed");
                      return;
                    }
                    await loadDeps();
                  }}
                  onShowQR={(departmentId) => {
                    setQrCodeType("department");
                    setShowQRCode(departmentId);
                  }}
                  t={t}
                  onDelete={async () => {
                    setDialogError(null);
                    setConfirmAction(() => async () => {
                      const r = await fetch(`/api/departments/${d.id}/delete`, {
                        method: "DELETE",
                      });
                      const j = await r.json();
                      if (!j.ok) {
                        setDialogError(j.error || "Delete failed");
                        throw new Error(j.error || "Delete failed");
                      }
                      if (depId === d.id) setDepId("");
                      await loadDeps();
                    });
                    setConfirmOpen(true);
                  }}
                />
              ))}
            </div>
          </div>
        )}
      </section>

      {/* Locations */}
      <section className='card p-5'>
        <div className='flex flex-col sm:flex-row sm:items-center justify-between gap-3'>
          <h2 className='text-lg font-semibold'>
            {t("admin.locations") || "Locations"}
          </h2>
          <div className='flex flex-col sm:flex-row sm:items-center gap-2 w-full sm:w-auto'>
            <Label className='text-sm'>{t("admin.departmentLabel")}:</Label>
            <Select value={depId} onValueChange={(value) => setDepId(value)}>
              <SelectTrigger className='sm:w-64 w-full'>
                <SelectValue placeholder={t("admin.selectDepartment")} />
              </SelectTrigger>
              <SelectContent>
                {deps.map((d) => (
                  <SelectItem key={d.id} value={d.id}>
                    {d.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Search locations */}
        {depId && (
          <div className='mt-4'>
            <div className='relative max-w-sm w-full'>
              <Search className='absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-foreground/50 z-10' />
              <Input
                type='text'
                placeholder={
                  t("admin.searchLocations") || "Search locations..."
                }
                value={locationSearchTerm}
                onChange={(e) => setLocationSearchTerm(e.target.value)}
                className='w-full pl-10'
              />
            </div>
          </div>
        )}

        {/* Add location */}
        <div className='mt-4'>
          <Dialog open={addLocationOpen} onOpenChange={setAddLocationOpen}>
            <DialogTrigger asChild>
              <Button disabled={!depId}>
                <Plus className='w-4 h-4' />
                {t("admin.addNewLocation") || "Add Location"}
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>
                  {currentDep
                    ? `${t("admin.addLocationTo") || "Add location to"} "${
                        currentDep.name
                      }"`
                    : t("admin.addNewLocation") || "Add Location"}
                </DialogTitle>
              </DialogHeader>
              <div className='py-4'>
                <div className='space-y-4'>
                  <div className='space-y-2'>
                    <Label htmlFor='locName'>
                      {t("admin.locationName") || "Location Name"}
                    </Label>
                    <Input
                      id='locName'
                      placeholder={
                        t("admin.enterLocationName") || "Enter location name"
                      }
                      value={locName}
                      onChange={(e) => setLocName(e.target.value)}
                      onKeyPress={(e) => {
                        if (e.key === "Enter") {
                          e.preventDefault();
                          handleAddLocation();
                        }
                      }}
                    />
                  </div>
                  <div className='flex justify-end gap-2'>
                    <Button
                      variant='outline'
                      onClick={() => {
                        setAddLocationOpen(false);
                        setLocName("");
                      }}
                    >
                      {t("common.cancel")}
                    </Button>
                    <Button
                      onClick={handleAddLocation}
                      disabled={!locName.trim() || !depId}
                    >
                      {t("admin.addNewLocation") || "Add Location"}
                    </Button>
                  </div>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>

        {/* List */}
        {!depId ? (
          <p className='mt-4 text-muted-foreground'>
            {t("admin.selectDepartmentToSee")}
          </p>
        ) : locations.length === 0 ? (
          <p className='mt-4 text-muted-foreground'>
            {t("admin.noLocationsInDepartment") ||
              "No locations in this department"}
          </p>
        ) : (
          <div className='mt-6'>
            <div className='flex flex-col sm:flex-row sm:items-center justify-between mb-4 gap-2'>
              <h3 className='text-lg font-semibold'>
                {t("admin.locationsWithCount") || "Locations"} (
                {locations.length})
              </h3>
              <p className='text-sm text-muted-foreground text-center sm:text-right'>
                {t("ticketDetail.tapLocationToManage") ||
                  "Tap a location to manage its elements"}
              </p>
            </div>
            <div className='grid grid-cols-1 sm:grid-cols-2 xl:grid-cols-3 gap-4 sm:gap-6'>
              {locations.map((loc) => (
                <LocationRow
                  key={loc.id}
                  location={loc}
                  selected={locationId === loc.id}
                  onSelect={() => setLocationId(loc.id)}
                  onRename={async (newName) => {
                    const r = await fetch(`/api/locations/${loc.id}/update`, {
                      method: "PATCH",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({ name: newName }),
                    });
                    const j = await r.json();
                    if (!j.ok) {
                      setErr(j.error || "Update failed");
                      return;
                    }
                    await loadLocations(depId);
                  }}
                  onToggle={async () => {
                    const r = await fetch(`/api/locations/${loc.id}/toggle`, {
                      method: "PATCH",
                    });
                    const j = await r.json();
                    if (!j.ok) {
                      setErr(j.error || "Toggle failed");
                      return;
                    }
                    await loadLocations(depId);
                  }}
                  onShowQR={(locationId) => {
                    setQrCodeType("location");
                    setShowQRCode(locationId);
                  }}
                  onDelete={() => {
                    setConfirmAction(() => async () => {
                      const r = await fetch(`/api/locations/${loc.id}/delete`, {
                        method: "DELETE",
                      });
                      const j = await r.json();
                      if (!j.ok) {
                        setErr(j.error || "Delete failed");
                        return;
                      }
                      await loadLocations(depId);
                      if (locationId === loc.id) {
                        setLocationId("");
                      }
                    });
                    setConfirmOpen(true);
                  }}
                  t={t}
                />
              ))}
            </div>
          </div>
        )}
      </section>

      {/* Elements */}
      <section className='card p-5'>
        <div className='flex flex-col sm:flex-row sm:items-center justify-between gap-3'>
          <h2 className='text-lg font-semibold'>
            {t("admin.elements") || "Equipment"}
          </h2>
          <div className='flex flex-col sm:flex-row sm:items-center gap-2 w-full sm:w-auto'>
            <Label className='text-sm'>
              {t("admin.locationLabel") || "Location"}:
            </Label>
            <Select
              value={locationId}
              onValueChange={(value) => setLocationId(value)}
              disabled={!depId}
            >
              <SelectTrigger className='sm:w-64 w-full'>
                <SelectValue
                  placeholder={t("admin.selectLocation") || "Select location"}
                />
              </SelectTrigger>
              <SelectContent>
                {locations.map((l) => (
                  <SelectItem key={l.id} value={l.id}>
                    {l.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Search elements */}
        {locationId && (
          <div className='mt-4'>
            <div className='relative max-w-sm w-full'>
              <Search className='absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-foreground/50 z-10' />
              <Input
                type='text'
                placeholder={t("admin.searchElements")}
                value={elementSearchTerm}
                onChange={(e) => setElementSearchTerm(e.target.value)}
                className='w-full pl-10'
              />
            </div>
          </div>
        )}

        {/* Add element */}
        <div className='mt-4'>
          <Dialog open={addElementOpen} onOpenChange={setAddElementOpen}>
            <DialogTrigger asChild>
              <Button disabled={!locationId}>
                <Plus className='w-4 h-4' />
                {t("admin.addNewElement") || "Add Equipment"}
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>
                  {currentLoc
                    ? `${t("admin.addElementTo") || "Add equipment to"} "${
                        currentLoc.name
                      }"`
                    : t("admin.addNewElement") || "Add Equipment"}
                </DialogTitle>
              </DialogHeader>
              <div className='py-4'>
                <div className='space-y-4'>
                  <div className='space-y-2'>
                    <Label htmlFor='elName'>
                      {t("admin.elementName") || "Equipment Name"}
                    </Label>
                    <Input
                      id='elName'
                      placeholder={
                        t("admin.enterElementName") || "Enter equipment name"
                      }
                      value={elName}
                      onChange={(e) => setElName(e.target.value)}
                      onKeyPress={(e) => {
                        if (e.key === "Enter") {
                          e.preventDefault();
                          handleAddElement();
                        }
                      }}
                    />
                  </div>
                  <div className='flex justify-end gap-2'>
                    <Button
                      variant='outline'
                      onClick={() => {
                        setAddElementOpen(false);
                        setElName("");
                      }}
                    >
                      {t("common.cancel")}
                    </Button>
                    <Button
                      onClick={handleAddElement}
                      disabled={!elName.trim() || !locationId}
                    >
                      {t("admin.addNewElement") || "Add Equipment"}
                    </Button>
                  </div>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>

        {/* List */}
        {!locationId ? (
          <p className='mt-4 text-muted-foreground'>
            {t("admin.selectLocationToSee") ||
              "Select a location to see equipment"}
          </p>
        ) : elements.length === 0 ? (
          <p className='mt-4 text-muted-foreground'>
            {t("admin.noElementsInLocation") || "No equipment in this location"}
          </p>
        ) : (
          <div className='mt-4 space-y-4'>
            {elements.map((el) => (
              <ElementRow
                key={el.id}
                item={el}
                locations={locations}
                roles={roles}
                expanded={expandedElements.has(el.id)}
                onToggleExpansion={() => {
                  const set = new Set(expandedElements);
                  set.has(el.id) ? set.delete(el.id) : set.add(el.id);
                  setExpandedElements(set);
                }}
                onRename={async (v) => {
                  const r = await fetch(`/api/elements/${el.id}/update`, {
                    method: "PATCH",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ name: v }),
                  });
                  const j = await r.json();
                  if (!j.ok) {
                    setErr(j.error || "Update failed");
                    return;
                  }
                  await loadElements(locationId);
                }}
                onMove={async (newLocId) => {
                  const r = await fetch(`/api/elements/${el.id}/update`, {
                    method: "PATCH",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ locationId: newLocId }),
                  });
                  const j = await r.json();
                  if (!j.ok) {
                    setErr(j.error || "Move failed");
                    return;
                  }
                  await loadElements(locationId);
                }}
                onToggle={async () => {
                  const r = await fetch(`/api/elements/${el.id}/toggle`, {
                    method: "PATCH",
                  });
                  const j = await r.json();
                  if (!j.ok) {
                    setErr(j.error || "Toggle failed");
                    return;
                  }
                  await loadElements(locationId);
                }}
                onDelete={() => {
                  setConfirmAction(() => async () => {
                    const r = await fetch(`/api/elements/${el.id}/delete`, {
                      method: "DELETE",
                    });
                    const j = await r.json();
                    if (!j.ok) {
                      setErr(j.error || "Delete failed");
                      return;
                    }
                    await loadElements(locationId);
                  });
                  setConfirmOpen(true);
                }}
                onAssignRole={async (eid, rid) => {
                  const r = await fetch(`/api/admin/roles/${rid}/elements`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ elementId: eid }),
                  });
                  const j = await r.json();
                  if (!j.ok) throw new Error(j.error);
                }}
                onRemoveRole={async (eid, rid) => {
                  const r = await fetch(`/api/admin/roles/${rid}/elements`, {
                    method: "DELETE",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({ elementId: eid }),
                  });
                  const j = await r.json();
                  if (!j.ok) throw new Error(j.error);
                }}
                getElementRoles={async (eid) => {
                  const r = await fetch(`/api/elements/${eid}/roles`);
                  const j = await r.json();
                  return j.roles || [];
                }}
                onShowQR={(elementId) => {
                  setQrCodeType("element");
                  setShowQRCode(elementId);
                }}
                currentLocationId={locationId}
                currentDepartmentId={depId}
                t={t}
              />
            ))}
          </div>
        )}
      </section>

      {/* Confirm Dialog */}
      <ConfirmDialog
        open={confirmOpen}
        onClose={() => {
          setConfirmOpen(false);
          setDialogError(null);
        }}
        onConfirm={async () => {
          if (confirmAction) {
            await confirmAction();
          }
        }}
        title={t("admin.confirmDeletion") || "Please confirm"}
        message={t("admin.cannotBeUndone") || "This action cannot be undone."}
        confirmText={t("common.confirm") || "Yes, proceed"}
        cancelText={t("common.cancel") || "Cancel"}
        type='error'
        errorMessage={dialogError ?? undefined}
      />

      {/* QR Modal */}
      {showQRCode && (
        <div
          className='fixed inset-0 bg-black/50 flex items-center justify-center z-50'
          data-modal='qr-code'
        >
          <div className='bg-card rounded-lg p-4 sm:p-6 w-[95%] max-w-md mx-auto'>
            <div className='flex items-center justify-between mb-4'>
              <h3 className='text-lg font-semibold'>
                {t("admin.qrCodeFor")}{" "}
                {qrCodeType === "department"
                  ? deps.find((d) => d.id === showQRCode)?.name
                  : qrCodeType === "location"
                  ? locations.find((l) => l.id === showQRCode)?.name
                  : elements.find((e) => e.id === showQRCode)?.name}
              </h3>
              <button
                onClick={() => setShowQRCode(null)}
                className='text-muted-foreground hover:text-foreground'
              >
                <X className='w-5 h-5' />
              </button>
            </div>
            <div className='flex flex-col items-center space-y-4'>
              <QRCodeGenerator
                url={generateQRUrl(
                  showQRCode,
                  qrCodeType,
                  qrCodeType === "location"
                    ? locations.find((l) => l.id === showQRCode)?.departmentId
                    : qrCodeType === "element"
                    ? (() => {
                        const element = elements.find(
                          (e) => e.id === showQRCode
                        );
                        if (element?.locationId) {
                          const loc = locations.find(
                            (l) => l.id === element.locationId
                          );
                          return loc?.departmentId;
                        }
                        return undefined;
                      })()
                    : undefined,
                  qrCodeType === "element"
                    ? elements.find((e) => e.id === showQRCode)?.locationId
                    : undefined
                )}
                size={200}
                className='border-2 border-muted rounded-lg'
              />
              <p className='text-sm text-muted-foreground text-center'>
                {t("admin.scanQRCodeToCreateTicket")}{" "}
                {qrCodeType === "department"
                  ? t("admin.department")
                  : qrCodeType === "location"
                  ? t("admin.location")
                  : t("ticketDetail.equipment") || "equipment"}
              </p>
              <div className='flex flex-col sm:flex-row gap-2 w-full'>
                <button
                  onClick={() =>
                    downloadQRCode(
                      showQRCode,
                      qrCodeType === "department"
                        ? deps.find((d) => d.id === showQRCode)?.name || ""
                        : qrCodeType === "location"
                        ? locations.find((l) => l.id === showQRCode)?.name || ""
                        : elements.find((e) => e.id === showQRCode)?.name || "",
                      qrCodeType
                    )
                  }
                  className='btn-secondary inline-flex items-center justify-center gap-2 w-full sm:w-auto'
                >
                  <Download className='w-4 h-4' />
                  {t("admin.download")}
                </button>
                <button
                  onClick={() => setShowQRCode(null)}
                  className='btn-primary w-full sm:w-auto'
                >
                  {t("common.close")}
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </main>
  );
}

/* Department row with inline edit */
function DepartmentRow({
  dept,
  selected,
  onSelect,
  onRename,
  onToggle,
  onDelete,
  onShowQR,
  t,
}: {
  dept: Department;
  selected: boolean;
  onSelect: () => void;
  onRename: (v: string) => Promise<void>;
  onToggle: () => Promise<void>;
  onDelete: () => void;
  onShowQR: (departmentId: string) => void;
  t: (key: string) => string;
}) {
  const [editing, setEditing] = useState(false);
  const [value, setValue] = useState(dept.name);
  useEffect(() => setValue(dept.name), [dept.name]);

  return (
    <div
      className={`group relative overflow-hidden rounded-xl border transition-all duration-200 hover:shadow-md hover:-translate-y-0.5 ${
        selected
          ? "ring-2 ring-primary/30 bg-primary/5 border-primary/30 shadow-md"
          : "bg-card border-border hover:border-primary/30 hover:shadow-sm"
      } ${!dept.isVisible ? "opacity-60" : ""}`}
    >
      {/* Header Section */}
      <div className='relative p-4 pb-3'>
        {/* Status Badge */}
        <div className='absolute top-3 right-3'>
          <div
            className={`w-2 h-2 rounded-full ${
              dept.isVisible ? "bg-green-500" : "bg-gray-400"
            }`}
            title={dept.isVisible ? t("admin.active") : t("admin.disabled")}
          />
        </div>

        {/* Department Name */}
        <div className='pr-8'>
          <button
            onClick={onSelect}
            className='w-full text-left group-hover:text-primary transition-colors'
            title={t("admin.selectDepartment")}
          >
            <h3 className='text-lg font-semibold text-foreground group-hover:text-primary transition-colors'>
              {dept.name}
            </h3>
            {!dept.isVisible && (
              <span className='text-xs text-muted-foreground mt-1'>
                {t("admin.disabled")}
              </span>
            )}
          </button>
        </div>
      </div>

      {/* Action Buttons Section */}
      <div className='px-4 pb-4'>
        <div className='grid grid-cols-2 gap-1.5'>
          {editing ? (
            <div className='col-span-2 space-y-2'>
              <input
                value={value}
                onChange={(e) => setValue(e.target.value)}
                className='w-full px-3 py-2 text-sm border border-primary/30 rounded-lg bg-card text-foreground focus:outline-none focus:ring-2 focus:ring-primary/20 focus:border-primary transition-all'
                placeholder={t("admin.enterNewName")}
                autoFocus
              />
              <div className='flex gap-1.5'>
                <button
                  className='flex-1 inline-flex items-center justify-center gap-1.5 px-3 py-2 text-xs font-medium bg-green-600 text-white rounded-lg hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-green-200 transition-all'
                  onClick={async () => {
                    const v = value.trim();
                    if (!v || v === dept.name) {
                      setEditing(false);
                      return;
                    }
                    await onRename(v);
                    setEditing(false);
                  }}
                >
                  <Check className='w-3 h-3' /> {t("admin.save")}
                </button>
                <button
                  className='flex-1 inline-flex items-center justify-center gap-1.5 px-3 py-2 text-xs font-medium bg-gray-500 text-white rounded-lg hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-gray-200 transition-all'
                  onClick={() => {
                    setEditing(false);
                    setValue(dept.name);
                  }}
                >
                  <X className='w-3 h-3' /> {t("common.cancel")}
                </button>
              </div>
            </div>
          ) : (
            <>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-blue-50 text-blue-700 rounded-lg hover:bg-blue-100 focus:outline-none focus:ring-2 focus:ring-blue-200 transition-all border border-blue-200'
                onClick={() => setEditing(true)}
                title={t("admin.renameDepartment")}
              >
                <Pencil className='w-3 h-3' />
                <span>{t("admin.rename")}</span>
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-orange-50 text-orange-700 rounded-lg hover:bg-orange-100 focus:outline-none focus:ring-2 focus:ring-orange-200 transition-all border border-orange-200'
                onClick={onToggle}
                title={
                  dept.isVisible
                    ? t("admin.disableDepartment")
                    : t("admin.enableDepartment")
                }
              >
                {dept.isVisible ? (
                  <>
                    <EyeOff className='w-3 h-3' />
                    <span>{t("admin.disable")}</span>
                  </>
                ) : (
                  <>
                    <Eye className='w-3 h-3' />
                    <span>{t("admin.enable")}</span>
                  </>
                )}
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-purple-50 text-purple-700 rounded-lg hover:bg-purple-100 focus:outline-none focus:ring-2 focus:ring-purple-200 transition-all border border-purple-200'
                onClick={() => onShowQR(dept.id)}
                title={t("admin.generateQRCode")}
              >
                <QrCode className='w-3 h-3' />
                <span>{t("admin.qr") || "QR"}</span>
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-red-50 text-red-700 rounded-lg hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-red-200 transition-all border border-red-200'
                onClick={onDelete}
                title={t("admin.deleteDepartment")}
              >
                <Trash2 className='w-3 h-3' />
                <span>{t("common.delete")}</span>
              </button>
            </>
          )}
        </div>
      </div>
    </div>
  );
}

/* Location row with inline edit */
function LocationRow({
  location,
  selected,
  onSelect,
  onRename,
  onToggle,
  onDelete,
  onShowQR,
  t,
}: {
  location: LocationItem;
  selected: boolean;
  onSelect: () => void;
  onRename: (v: string) => Promise<void>;
  onToggle: () => Promise<void>;
  onDelete: () => void;
  onShowQR: (locationId: string) => void;
  t: (key: string) => string;
}) {
  const [editing, setEditing] = useState(false);
  const [value, setValue] = useState(location.name);
  useEffect(() => setValue(location.name), [location.name]);

  return (
    <div
      className={`group relative overflow-hidden rounded-xl border transition-all duration-200 hover:shadow-md hover:-translate-y-0.5 ${
        selected
          ? "ring-2 ring-primary/30 bg-primary/5 border-primary/30 shadow-md"
          : "bg-card border-border hover:border-primary/30 hover:shadow-sm"
      } ${!location.isVisible ? "opacity-60" : ""}`}
    >
      {/* Header Section */}
      <div className='relative p-4 pb-3'>
        {/* Status Badge */}
        <div className='absolute top-3 right-3'>
          <div
            className={`w-2 h-2 rounded-full ${
              location.isVisible ? "bg-green-500" : "bg-gray-400"
            }`}
            title={location.isVisible ? t("admin.active") : t("admin.disabled")}
          />
        </div>

        {/* Location Name */}
        <div className='pr-8'>
          <button
            onClick={onSelect}
            className='w-full text-left group-hover:text-primary transition-colors'
            title={t("admin.selectLocation")}
          >
            <h3 className='text-lg font-semibold text-foreground group-hover:text-primary transition-colors'>
              {location.name}
            </h3>
            {!location.isVisible && (
              <span className='text-xs text-muted-foreground mt-1'>
                {t("admin.disabled")}
              </span>
            )}
          </button>
        </div>
      </div>

      {/* Action Buttons Section */}
      <div className='px-4 pb-4'>
        <div className='grid grid-cols-2 gap-1.5'>
          {editing ? (
            <div className='col-span-2 space-y-2'>
              <input
                value={value}
                onChange={(e) => setValue(e.target.value)}
                className='w-full px-3 py-2 text-sm border border-primary/30 rounded-lg bg-card text-foreground focus:outline-none focus:ring-2 focus:ring-primary/20 focus:border-primary transition-all'
                placeholder={t("admin.enterNewName")}
                autoFocus
              />
              <div className='flex gap-1.5'>
                <button
                  className='flex-1 inline-flex items-center justify-center gap-1.5 px-3 py-2 text-xs font-medium bg-green-600 text-white rounded-lg hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-green-200 transition-all'
                  onClick={async () => {
                    const v = value.trim();
                    if (!v || v === location.name) {
                      setEditing(false);
                      return;
                    }
                    await onRename(v);
                    setEditing(false);
                  }}
                >
                  <Check className='w-3 h-3' /> {t("admin.save")}
                </button>
                <button
                  className='flex-1 inline-flex items-center justify-center gap-1.5 px-3 py-2 text-xs font-medium bg-gray-500 text-white rounded-lg hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-gray-200 transition-all'
                  onClick={() => {
                    setEditing(false);
                    setValue(location.name);
                  }}
                >
                  <X className='w-3 h-3' /> {t("common.cancel")}
                </button>
              </div>
            </div>
          ) : (
            <>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-blue-50 text-blue-700 rounded-lg hover:bg-blue-100 focus:outline-none focus:ring-2 focus:ring-blue-200 transition-all border border-blue-200'
                onClick={() => setEditing(true)}
                title={t("admin.renameLocation")}
              >
                <Pencil className='w-3 h-3' />
                <span>{t("admin.rename")}</span>
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-orange-50 text-orange-700 rounded-lg hover:bg-orange-100 focus:outline-none focus:ring-2 focus:ring-orange-200 transition-all border border-orange-200'
                onClick={onToggle}
                title={
                  location.isVisible
                    ? t("admin.disableLocation")
                    : t("admin.enableLocation")
                }
              >
                {location.isVisible ? (
                  <>
                    <EyeOff className='w-3 h-3' />
                    <span>{t("admin.disable")}</span>
                  </>
                ) : (
                  <>
                    <Eye className='w-3 h-3' />
                    <span>{t("admin.enable")}</span>
                  </>
                )}
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-purple-50 text-purple-700 rounded-lg hover:bg-purple-100 focus:outline-none focus:ring-2 focus:ring-purple-200 transition-all border border-purple-200'
                onClick={() => onShowQR(location.id)}
                title={t("admin.generateQRCode")}
              >
                <QrCode className='w-3 h-3' />
                <span>{t("admin.qr") || "QR"}</span>
              </button>
              <button
                className='group/btn inline-flex items-center justify-center gap-1.5 px-2.5 py-2 text-xs font-medium bg-red-50 text-red-700 rounded-lg hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-red-200 transition-all border border-red-200'
                onClick={onDelete}
                title={t("admin.deleteLocation")}
              >
                <Trash2 className='w-3 h-3' />
                <span>{t("common.delete")}</span>
              </button>
            </>
          )}
        </div>
      </div>
    </div>
  );
}

/* Element row with inline rename + move + delete + role management */
function ElementRow({
  item,
  locations,
  roles,
  expanded,
  onToggleExpansion,
  onRename,
  onMove,
  onToggle,
  onDelete,
  onAssignRole,
  onRemoveRole,
  getElementRoles,
  onShowQR,
  currentLocationId,
  currentDepartmentId,
  t,
}: {
  item: ElementItem;
  locations: LocationItem[];
  roles: Role[];
  expanded: boolean;
  onToggleExpansion: () => void;
  onRename: (v: string) => Promise<void>;
  onMove: (locId: string) => Promise<void>;
  onToggle: () => Promise<void>;
  onDelete: () => void;
  onAssignRole: (elementId: string, roleId: string) => Promise<any>;
  onRemoveRole: (elementId: string, roleId: string) => Promise<void>;
  getElementRoles: (elementId: string) => Promise<ElementRole[]>;
  onShowQR: (elementId: string) => void;
  currentLocationId?: string;
  currentDepartmentId?: string;
  t: (key: string) => string;
}) {
  const [name, setName] = useState(item.name);
  const [elementRoles, setElementRoles] = useState<ElementRole[]>([]);
  const [loadingRoles, setLoadingRoles] = useState(false);
  const [selectedRoleId, setSelectedRoleId] = useState("");

  useEffect(() => setName(item.name), [item.name]);

  useEffect(() => {
    if (expanded) {
      loadElementRoles();
    }
  }, [expanded, item.id]);

  const loadElementRoles = async () => {
    setLoadingRoles(true);
    try {
      const roles = await getElementRoles(item.id);
      setElementRoles(roles);
    } catch (e) {
      console.error("Failed to load element roles:", e);
    } finally {
      setLoadingRoles(false);
    }
  };

  const handleAssignRole = async () => {
    if (!selectedRoleId) return;
    try {
      await onAssignRole(item.id, selectedRoleId);
      setSelectedRoleId("");
      await loadElementRoles();
    } catch (e) {
      // Error already handled in parent
    }
  };

  const handleRemoveRole = async (roleId: string) => {
    try {
      await onRemoveRole(item.id, roleId);
      await loadElementRoles();
    } catch (e) {
      // Error already handled in parent
    }
  };

  const availableRoles = roles.filter(
    (role) => !elementRoles.some((er) => er.id === role.id)
  );

  return (
    <div className={`${!item.isVisible ? "opacity-60" : ""}`}>
      {/* Main element row */}
      <div className='p-3 flex flex-col lg:flex-row lg:items-center gap-2'>
        <Input
          value={name}
          onChange={(e) => setName(e.target.value)}
          onBlur={async () => {
            const v = name.trim();
            if (v && v !== item.name) await onRename(v);
          }}
          className='flex-1 w-full lg:w-auto'
          placeholder={
            item.name +
            (!item.isVisible ? ` ${t("admin.elementDisabled")}` : "")
          }
        />
        <Select
          value={item.locationId}
          onValueChange={async (newLoc) => {
            if (newLoc && newLoc !== item.locationId) await onMove(newLoc);
          }}
        >
          <SelectTrigger
            className='min-w-[150px] w-full lg:w-auto'
            title={t("admin.moveElementToLocation")}
          >
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            {locations.map((l) => (
              <SelectItem key={l.id} value={l.id}>
                {l.name}
              </SelectItem>
            ))}
          </SelectContent>
        </Select>
        <div className='flex flex-wrap gap-2 mt-2 lg:mt-0'>
          <Button
            variant='outline'
            className='flex-1 sm:flex-none whitespace-nowrap text-sm'
            onClick={onToggle}
            title={
              item.isVisible
                ? t("admin.disableElement")
                : t("admin.enableElement")
            }
          >
            {item.isVisible ? (
              <>
                <EyeOff className='w-4 h-4' />{" "}
                <span className='hidden sm:inline'>{t("admin.disable")}</span>
              </>
            ) : (
              <>
                <Eye className='w-4 h-4' />{" "}
                <span className='hidden sm:inline'>{t("admin.enable")}</span>
              </>
            )}
          </Button>
          <Button
            variant='outline'
            className='flex-1 sm:flex-none whitespace-nowrap text-sm'
            onClick={onToggleExpansion}
            title={t("admin.manageRolesForElement")}
          >
            <Users className='w-4 h-4' />
            {expanded ? (
              <>
                <ChevronDown className='w-4 h-4' />{" "}
                <span className='hidden sm:inline'>{t("admin.roles")}</span>
              </>
            ) : (
              <>
                <ChevronRight className='w-4 h-4' />{" "}
                <span className='hidden sm:inline'>{t("admin.roles")}</span>
              </>
            )}
          </Button>
          <Button
            variant='outline'
            className='flex-1 sm:flex-none whitespace-nowrap text-sm'
            onClick={() => onShowQR(item.id)}
            title={t("admin.generateQRCode")}
          >
            <QrCode className='w-4 h-4' />
            <span className='hidden sm:inline'>{t("admin.qr") || "QR"}</span>
          </Button>
          <Button
            variant='outline'
            className='flex-1 sm:flex-none whitespace-nowrap text-sm'
            onClick={onDelete}
          >
            <Trash2 className='w-4 h-4' />{" "}
            <span className='hidden sm:inline'>{t("common.delete")}</span>
          </Button>
        </div>
      </div>

      {/* Expanded role management section */}
      {expanded && (
        <div className='px-3 pb-3 bg-gray-50 border-t border-gray-200'>
          <div className='pt-3 space-y-3'>
            <h4 className='font-medium text-sm text-gray-700'>
              {t("admin.roleAssignmentsFor")} {item.name}
            </h4>

            {/* Assigned Roles */}
            <div>
              <h5 className='text-xs font-medium text-green-700 mb-2'>
                {t("admin.assignedRoles")} ({elementRoles.length})
              </h5>
              {loadingRoles ? (
                <div className='flex items-center gap-2 text-sm text-gray-500'>
                  <Loader2 className='w-4 h-4 animate-spin' />
                  {t("admin.loadingRoles")}
                </div>
              ) : elementRoles.length > 0 ? (
                <div className='grid grid-cols-1 md:grid-cols-2 gap-2'>
                  {elementRoles.map((role) => (
                    <div
                      key={role.id}
                      className='flex items-center justify-between p-2 bg-green-50 border border-green-200 rounded'
                    >
                      <div>
                        <span className='font-medium text-sm'>{role.name}</span>
                        {role.description && (
                          <span className='text-xs text-gray-500 ml-2'>
                            ({role.description})
                          </span>
                        )}
                      </div>
                      <button
                        onClick={() => handleRemoveRole(role.id)}
                        className='text-red-600 hover:text-red-700 text-xs'
                      >
                        <X className='w-3 h-3' />
                      </button>
                    </div>
                  ))}
                </div>
              ) : (
                <p className='text-sm text-gray-500'>
                  {t("admin.noRolesAssigned")}
                </p>
              )}
            </div>

            {/* Assign New Role */}
            <div>
              <h5 className='text-xs font-medium text-blue-700 mb-2'>
                {t("admin.assignNewRole")}
              </h5>
              <div className='flex items-center gap-2'>
                <Select
                  value={selectedRoleId}
                  onValueChange={setSelectedRoleId}
                >
                  <SelectTrigger className='text-sm'>
                    <SelectValue placeholder={t("admin.selectRole")} />
                  </SelectTrigger>
                  <SelectContent>
                    {availableRoles.map((role) => (
                      <SelectItem key={role.id} value={role.id}>
                        {role.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button
                  onClick={handleAssignRole}
                  disabled={!selectedRoleId}
                  size='sm'
                  className='text-xs px-3'
                >
                  <Plus className='w-3 h-3' />
                  {t("admin.assign")}
                </Button>
              </div>
              {availableRoles.length === 0 && (
                <p className='text-xs text-gray-500 mt-1'>
                  {t("admin.allRolesAssigned")}
                </p>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
